<?php
require_once(dirname(__FILE__) . "/../../includes/DBConnection.php");
require_once(dirname(__FILE__) . "/../../core/truck_owner/truck_owner_core.php");
require_once(dirname(__FILE__) . "/../../core/company/trucking_company/trucking_company_core.php");
require_once(dirname(__FILE__) . "/../../core/company/clearing_agent/clearing_agent_core.php");
require_once(dirname(__FILE__) . "/../../core/payment_agent/payment_agent_core.php");
require_once(dirname(__FILE__) . "/../../core/company/company_core.php");
require_once(dirname(__FILE__) . "/../../core/company_employee/company_employee_core.php");


class AccountCore
{

    private $_truckingCompanyCore;
    private $_clearingAgentCore;
    private $_paymentAgentCore;
    private $_companyCore;
    private $_companyEmployeeCore;

    public function __construct()
    {
        DBConnection::getInstance();
        $this->_truckingCompanyCore = new TruckingCompanyCore();
        $this->_paymentAgentCore = new PaymentAgentCore();
        $this->_clearingAgentCore = new ClearingAgentCore();
        $this->_companyCore = new CompanyCore();
        $this->_companyEmployeeCore = new CompanyEmployeeCore();
    }


    // --------------------------------------------------------------------------------------- //
    // --------------------- get info for a certian sub account without activities ----------- //
    // -------------------------------------------------------------------------------------- //
    public function getAccountAuthority($session)
    {   
        // if session id for corporate , inject the account_ids filter
        if (strtoupper($session['type']) == "CORPORATE") {
            // get tender_ids from company employee
            $company_employee_id = $session['company_employee_id'];
            $companyEmployeeBean = $this->_companyEmployeeCore->getCompanyEmployeeBasic($company_employee_id, 0);          
            if ($companyEmployeeBean->account_ids) {
                if ($companyEmployeeBean->account_ids == ['*']) {
                    return false;
                }
                return $companyEmployeeBean->account_ids;
            } else {
                throw new Exception("لا يوجد لديك صلاحية لمشاهدة الحسابات");
            }
        }else{
            return false;
        }
    }


    // ---------------------------------------------------------------------- //
    // --------------------- get info for a certian account ----------------- //
    // ---------------------------------------------------------------------- //
    public function getAccount($id, $user_id)
    {

        // if the id in the format X-X , then replace - with ,
        if (strpos($id, '-') == true) {
            $id = str_replace("-", ",", $id);
        }

        // if account_id is only one number without "," then add 0 as sub account id
        if (strpos($id, ',') == false) {
            $id = $id . ",0";
        }

        // if the account-id is not enclosed in single quote
        if (strpos($id, '\'') == false) {
            $id = "'$id'";
        }

        $accountInfo = DBConnection::getObjectBean("account", $id, $user_id);
        return $accountInfo;
    }

    // ----------------------------------------------------------------------------------- //
    // --------------------- get info for a certian account without activities ----------- //
    // ----------------------------------------------------------------------------------- //
    public function getAccountBasic($id, $user_id)
    {

        // if the id in the format X-X , then replace - with ,
        if (strpos($id, '-') == true) {
            $id = str_replace("-", ",", $id);
        }

        // if account_id is only one number without "," then add 0 as sub account id
        if (strpos($id, ',') == false) {
            $id = $id . ",0";
        }

        // if the account-id is not enclosed in single quote
        if (strpos($id, '\'') == false) {
            $id = "'$id'";
        }
        $accountInfo = DBConnection::getBasicObjectBean("account", $id, $user_id);
        return $accountInfo;
    }

    // --------------------------------------------------------------------------------------- //
    // --------------------- get info for a certian sub account without activities ----------- //
    // -------------------------------------------------------------------------------------- //
    public function getSubAccountBasic($id, $user_id)
    {
        $accountInfo = DBConnection::getBasicObjectBean("account", $id, $user_id);
        return $accountInfo;
    }


    // -------------------------------------------------------------------------- //
    // -------------------search for accounts using any search filter ----------- //
    // -------------------------------------------------------------------------- //
    public function searchAccount($searchFilter, $limit, $offset, $user_id, $orderBy = null)
    {

        $searchAccountResult = DBConnection::searchDB("account", $searchFilter, $limit, $offset, $user_id, $orderBy);
        return $searchAccountResult;
    }

    // ---------------------------------------------------------------------------- //
    // ------------------ get the account associated with the driver -------------- //
    // ---------------------------------------------------------------------------- //
    public function getDriverAccount($driver_id)
    {

        if (!$driver_id) {
            throw new Exception('لم يتمكن النظام من العثور على حساب السائق حيث أنه غير محدد على المستند', 555);
        }

        // get the user_id of the driver
        $driverCore = new DriverCore();
        $driverBean = $driverCore->getDriverBasic($driver_id, 0);
        $user_id = $driverBean->user_id;

        // search for the account
        $driverFilter = [['key' => "user_id", 'val' => $user_id], ['key' => "status", 'val' => 'ACTIVE']];
        $account_qry = $this->searchAccount($driverFilter, 1, 0, 0);

        if ($account_qry->found_rows == 0) {
            throw new Exception('لم يتمكن النظام من العثور على حساب السائق', 555);
        }

        return $account_qry->data[0]->id;
    }

    // --------------------------------------------------------------------------------- //
    // ------------------ get the account associated with the truck_owner -------------- //
    // --------------------------------------------------------------------------------- //
    public function getTruckOwnerAccount($truck_owner_id)
    {

        // get the user_id of the truck_owner
        $truckOwnerCore = new TruckOwnerCore();
        $truckOwnerBean = $truckOwnerCore->getTruckOwnerBasic($truck_owner_id, 0);
        $user_id = $truckOwnerBean->user_id;
        $company_id = $truckOwnerBean->company_id;

        // search for the account
        $driverFilter = [['key' => "user_id", 'val' => $user_id], ['key' => "status", 'val' => 'ACTIVE']];
        $account_qry = $this->searchAccount($driverFilter, 1, 0, 0);
        if ($account_qry->found_rows == 0) {
            if ($company_id) {
                // search for the account
                $driverFilter = [['key' => "company_id", 'val' => $company_id], ['key' => "status", 'val' => 'ACTIVE']];
                $account_qry = $this->searchAccount($driverFilter, 1, 0, 0);
                if ($account_qry->found_rows == 0) {
                    throw new Exception('لم يتمكن النظام من العثور على حساب المالك', 555);
                }
            } else {
                throw new Exception('لم يتمكن النظام من العثور على حساب المالك', 555);
            }
        }

        return $account_qry->data[0]->id;
    }


    // ---------------------------------------------------------------------------------------- //
    // --------------------- Create New Account ----------------------------------------------- //
    // ---------------------------------------------------------------------------------------- //
    public function createAccount($accountBean, $account_cat, $update_by)
    {

        try {

            $account = new stdClass();

            // generate account_id
            if ($account_cat == "user") {
                $accountId = $this->generateUserAccountId($accountBean->user_id);
                $company_type = "user";
                $account->type = "WLT";
            }
            // if the company is comapny
            else if ($account_cat == "company") {
                $accountId = $this->generateTruckingCompanyAccountId($accountBean->company_id);
                $company_type = "tc";
                $account->type = "PLC";
            }
            // if the company is payment_agent
            else if ($account_cat == "payment_agent") {
                $accountId = $this->generateCompanyAccountId();
                $company_type = "pa";
                $account->type = "PLC";
            }
            // if the company is clearing agent (Owner)
            else if ($account_cat == "owner") {
                $accountId = $this->generateCompanyAccountId();
                $company_type = "ca";
                $account->type = "PLC";
            }

            // prepare account data            
            $account->id = $accountId;
            $account->sub_id = 0;
            $account->company_type = $company_type;
            $account->name = $accountBean->name;
            $account->user_id = $accountBean->user_id;
            $account->company_id = $accountBean->company_id;
            $account->balance = 0;
            $account->minimum_balance = 0;
            $account->currency = 'JOD';
            $account->status = 'NEW';

            // create record
            $createAccountResult = DBConnection::insertDB("account", $account, 0);

            // activate it
            $updateStruct = new stdClass();
            $updateStruct->id =  $accountId . ",0";
            $updateStruct->status = 'ACTIVE';
            DBConnection::updateDB("account", $updateStruct, 0);


            // setup sub-accounts
            if ($account_cat != "user") {
                $this->setupSubAccounts($accountBean->company_id, 0);
            }

            return $accountId;
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }


    // ------------------------------------------------------------------------------------------------------------------------- //
    // ------------- Create sub-account dynamicaly when a new financial connection will be generated between 2 parties --------- //
    // ------------------------------------------------------------------------------------------------------------------------- //
    public function createSubAccount($account_id, $target_id, $account_type, $account_name, $user_id)
    {

        try {

            // get the place holder account bean
            if (gettype($account_id) == "object") {
                $accountBean = $account_id;
                $account_id = $accountBean->id;
                $accountBean = $this->getAccountBasic($account_id, $user_id);
            } else {
                $accountBean = $this->getAccountBasic($account_id, $user_id);
            }

            // generate the sub-account id using the company_id
            $subAccountId = $this->generateSubAccountId($account_type, $target_id, $accountBean->company_id);

            // prepare the sub-account bean
            $subAccount = new stdClass();
            $subAccount->id = $account_id;
            $subAccount->sub_id = $subAccountId;
            $subAccount->company_type = $accountBean->company_type;
            $subAccount->type = $account_type;
            $subAccount->name = $account_name;
            $subAccount->user_id = null;
            $subAccount->company_id = $accountBean->company_id;
            $subAccount->balance = 0;
            $subAccount->balance = 0;
            $subAccount->currency = 'JOD';
            $subAccount->status = 'NEW';

            // insert into Database
            $createAccountResult = DBConnection::insertDB("account", $subAccount, 0);

            // activate it
            $updateStruct = new stdClass();
            $updateStruct->id =  $account_id . "," . $subAccountId;
            $updateStruct->status = 'ACTIVE';

            DBConnection::updateDB("account", $updateStruct, 0);
            return $account_id . "," . $subAccountId;
        } catch (Exception $e) {
            if (strpos($e->getMessage(), "Integrity constraint violation: 1062 Duplicate entry")) {
                throw new Exception("الحساب مضاف مسبقا");
            } else {
                throw new Exception($e->getMessage());
            }
        }
    }


    // ----------------------------------------------------------------- //
    // --------------------- update account info ----------------------- //
    // ----------------------------------------------------------------- //
    public function updateAccount($accountBean, $account_id, $updated_by)
    {
        // fill update struct and set the target user id
        $accountBean->id = $account_id;
        unset($accountBean->balance);

        // get basic object bean without any activites
        $DB_Bean = DBConnection::getBasicObjectBean("account", "'$account_id'", $updated_by);

        //update account info only if the account bean is different than DB bean
        if (compareObject($accountBean, $DB_Bean) == false) {

            // map all the new values into the DB_bean
            $accountBean = mapBeanToDBBean($accountBean, $DB_Bean);
            unset($accountBean->balance);

            // fill update struct and set the target user id
            DBConnection::updateDB("account", $accountBean, $updated_by);
        }
    }

    // -------------------------------------------------------------------------------------------------- //
    // ----------------- get account statment for report "كشف الحسابات" --------------------------------- //
    // ----------------- params: account_id, from_date, to_date, limit ---------------------------------- //
    // -------------------------------------------------------------------------------------------------- //
    public function getAccountStatement($accountFilter, $limit, $user_id)
    {

        $trx_qry = DBConnection::searchReport("account", "02", $accountFilter, $limit, 0, $user_id, ' order by id ');
        return $trx_qry;
    }


    // --------------------------------------------------------------------------------------------------- //
    // ------------- Validate if the user has the right auth to view any financial report ---------------- //
    // --------------------------------------------------------------------------------------------------- //
    public function hasAuthToViewFinancialReport($tender_id, $report_name, $user_roles)
    {

        $tenderCore = new TenderCore();
        $man = $tenderCore->getTenderManifest($tender_id, 0);
        $reports = $man['reports'];

        $allowedToViewReport = false;
        foreach ($reports as $report) {
            foreach ($user_roles as $userRole) {
                foreach ($report['roles'] as $allowedRole) {
                    if ($allowedRole == $userRole) {
                        $allowedToViewReport = true;
                    }
                }
            }
        }


        return $allowedToViewReport;
    }


    // ----------------------------------------------------------------------------------------------- //
    // ------------------- Setup trucking company sub accounts --------------------------------------- //
    // ------------------- Wallet, Expense, CAP, IS and Revenue accounts ----------------------------- //
    // ----------------------------------------------------------------------------------------------- //
    public function setupSubAccounts($company_id, $user_id)
    {

        try {

            // get the trucking company place holder account
            $accountFilter = [['key' => "company_id", 'val' => $company_id], ['key' => "status", 'val' => ['INACTIVE'], 'op' => "not int"]];
            $accountQry = $this->searchAccount($accountFilter, 1, 0, $user_id);

            if ($accountQry->found_rows == 0)
                throw new Exception("cant create sub-accounts , Invalid company_id");

            $accountBean = $accountQry->data[0];

            // create the cash account
            $target_id = null;
            $account_type = "CASH";
            $account_name = "صندوق النقد";
            $this->createSubAccount($accountBean, $target_id, $account_type, $account_name, $user_id);

            // create the Revenue account
            $target_id = null;
            $account_type = "R";
            $account_name = "عوائد الصرف";
            $this->createSubAccount($accountBean, $target_id, $account_type, $account_name, $user_id);

            // create the Expense account
            $target_id = null;
            $account_type = "EX";
            $account_name = "تكاليف صرف المستند";
            $this->createSubAccount($accountBean, $target_id, $account_type, $account_name, $user_id);

            // create the Income Summary account
            $target_id = null;
            $account_type = "IS";
            $account_name = "ملخص الدخل";
            $this->createSubAccount($accountBean, $target_id, $account_type, $account_name, $user_id);

            // create the Capital account
            $target_id = null;
            $account_type = "CAP";
            $account_name = "رأس المال";
            $this->createSubAccount($accountBean, $target_id, $account_type, $account_name, $user_id);
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }


    // ---------------------------------------------------------------------------------------------- //
    // ------------------- Setup new connection between 2 parties (PA, TC) -------------------------- //
    // ------------------- the connection is an AR sub-account in PA ledger and --------------------- //
    // ------------------- AP sub-account in TC ledger , this operation occurs when------------------ //
    // ------------------- pa pays for a certain waybill  ------------------------------------------- //
    // ---------------------------------------------------------------------------------------------- //
    public function setupFinancialConections_type1($pa_id, $trucking_company_id)
    {

        try {
            DBConnection::startTransaction();
            $user_id = 0; // by system

            // get the pa and tc beans
            $paymentAgentBean = $this->_paymentAgentCore->getPaymentAgentBasic($pa_id, $user_id);
            $truckingCompanyBean = $this->_truckingCompanyCore->getTruckingCompanyBasic($trucking_company_id, $user_id);
            $companyBean = $this->_companyCore->getCompanyBasic($truckingCompanyBean->company_id, $user_id);

            // get the payment agent place holder account
            $paSearchFilter = [['key' => "company_id", 'val' => $paymentAgentBean->company_id], ['key' => "status", 'val' => 'ACTIVE']];
            $accountQry = $this->searchAccount($paSearchFilter, 1, 0, $user_id);
            if ($accountQry->found_rows == 0)
                throw new Exception("Invalid pa id");
            $accountBean1 = $accountQry->data[0];

            // get the trucking company place holder account
            $tcSearchFilter = [['key' => "company_id", 'val' => $truckingCompanyBean->company_id], ['key' => "status", 'val' => 'ACTIVE']];
            $accountQry = $this->searchAccount($tcSearchFilter, 1, 0, $user_id);

            if ($accountQry->found_rows == 0)
                throw new Exception("Invalid tc id");
            $accountBean2 = $accountQry->data[0];

            // Create مطالبات شركة النقل in payment agent ledger
            $account_id = $accountBean1->id;
            $target_id = $truckingCompanyBean->id;
            $account_type = "AR";
            $account_name = "مطالبات " . $companyBean->name;

            $this->createSubAccount($account_id, $target_id, $account_type, $account_name, 0);

            // Create مستحقات مكتب الصرف in trucking company ledger
            $account_id = $accountBean2->id;
            $target_id = $paymentAgentBean->id;
            $account_type = "AP";
            $account_name = "مستحقات " . $paymentAgentBean->name;
            $this->createSubAccount($account_id, $target_id, $account_type, $account_name, $user_id);

            DBConnection::commitTransaction();
        } catch (Exception $e) {
            DBConnection::rollBackTransaction();
            throw new Exception($e->getMessage());
        }
    }

    // ---------------------------------------------------------------------------------------------- //
    // ------------------- Setup new connection between 2 parties (TC, CA) -------------------------- //
    // ------------------- the connection is an AP sub-account in TC ledger and --------------------- //
    // ------------------- AR sub-account in CA ledger , this operation occurs when------------------ //
    // ------------------- tc wants money from cargo owner upon discharge a waybill------------------ //
    // ------------------- Note: all owners are clearing agents ------------------------------------- //
    // ---------------------------------------------------------------------------------------------- //
    public function setupFinancialConections_type2($trucking_company_id, $ca_id)
    {

        try {
            $user_id = 0; // by system

            // get the ca(owner) and tc beans
            $ownerBean = $this->_clearingAgentCore->getClearingAgentBasic($ca_id, $user_id);
            $truckingCompanyBean = $this->_truckingCompanyCore->getTruckingCompanyBasic($trucking_company_id, $user_id);
            $companyBean_tc = $this->_companyCore->getCompanyBasic($truckingCompanyBean->company_id, $user_id);
            $companyBean_ca = $this->_companyCore->getCompanyBasic($ownerBean->company_id, $user_id);

            // get the clearing agent (owner) place holder account
            $paSearchFilter = [['key' => "company_id", 'val' => $ownerBean->company_id], ['key' => "status", 'val' => 'ACTIVE']];
            $accountQry = $this->searchAccount($paSearchFilter, 1, 0, $user_id);

            if ($accountQry->found_rows == 0) {
                // create subholder account
                $accountBean = new stdClass();
                $accountBean->name =  $companyBean_ca->name;
                $accountBean->user_id = null;
                $accountBean->company_id =  $companyBean_ca->id;
                $accountBean->minimum_balance = 0;
                $accountBean1 = new stdClass();
                $accountBean1->id = $this->createAccount($accountBean, "owner", 0);       // owners are clearing agents
            } else {
                $accountBean1 = $accountQry->data[0];
            }

            // get the trucking company place holder account
            $tcSearchFilter = [
                ['key' => "company_id", 'val' => $truckingCompanyBean->company_id],
                ['key' => "type", 'val' => 'PLC'],
                ['key' => "status", 'val' => 'ACTIVE']
            ];

            $accountQry = $this->searchAccount($tcSearchFilter, 1, 0, $user_id);
            if ($accountQry->found_rows == 0)
                throw new Exception("Invalid tc id");
            $accountBean2 = $accountQry->data[0];

            // Create مستحقات شركة النقل in clearing agent (owner) ledger
            $account_id = $accountBean1->id;
            $target_id = $truckingCompanyBean->id;
            $account_type = "AP";
            $account_name = "مستحقات " . $companyBean_tc->name;
            $this->createSubAccount($account_id, $target_id, $account_type, $account_name, 0);

            // Create مطالبات مالك البضاعة in trucking company ledger
            $account_id = $accountBean2->id;
            $target_id = $ownerBean->id;
            $account_type = "AR";
            $account_name = "مطالبات " . $companyBean_ca->name;
            $sub_id = $this->createSubAccount($account_id, $target_id, $account_type, $account_name, $user_id);

            return $sub_id;
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    // ------------------------------------------------------------------------------------------------ //
    // ------------------- Generate sub account id based on account type ------------------------------ //
    // ------------------------------------------------------------------------------------------------ //
    public function generateSubAccountId($account_type, $target_id = null, $company_id)
    {

        $subAccountId = "";

        switch ($account_type) {
            case 'WLT':
            case 'CASH':
            case 'BANK':
            case 'ZC':
                $subAccountId = $this->generateWalletAccountId($company_id);
                break;
            case 'AR':
                if (!$target_id)
                    throw new Exception("Invalid target id to generate sub_id");
                $subAccountId = 2 . str_pad($target_id, 4, "0", STR_PAD_LEFT);
                break;
            case 'AP':
                if (!$target_id)
                    throw new Exception("Invalid target id to generate sub_id");
                $subAccountId = 3 . str_pad($target_id, 4, "0", STR_PAD_LEFT);
                break;
            case 'R':
                $subAccountId = 50001;
                break;
            case 'EX':
                $subAccountId = 60001;
                break;
            case 'IS':
                $subAccountId = 70001;
                break;
            case 'CAP':
                $subAccountId = 70002;
                break;
        }

        return $subAccountId;
    }


    // ------------------------------------------------------------------------------------------------- //
    // --------------------- Generate uniqe account id to be given to the user ------------------------- //
    // ------------------------------------------------------------------------------------------------- //
    private function generateWalletAccountId($company_id)
    {
        $sqlQuery = " SELECT
                        MAX(t.sub_id) + 1 account_id
                      FROM
                        account t
                      WHERE
                        type in('WLT','BANK','CASH','ZC') AND company_id = ?";

        $param = [$company_id];
        $result = DBConnection::runBindDatabaseQuery($sqlQuery, $param);

        if ($result[0]->account_id) {
            return $result[0]->account_id;
        } else {
            return 19901;
        }
    }

    // ------------------------------------------------------------------------------------------------- //
    // --------------------- Generate uniqe account id to be given to the user ------------------------- //
    // ------------------------------------------------------------------------------------------------- //
    private function generateUserAccountId($user_id)
    {
        $sqlQuery = " SELECT
                        DISTINCT (t.id + 530000) account_id
                    FROM
                        user t
                    where id = ?";

        $param = [$user_id];
        $result = DBConnection::runBindDatabaseQuery($sqlQuery, $param);

        if ($result[0]) {
            return $result[0]->account_id;
        } else {
            throw new Exception('لا يمكن انشاء حساب جديد للمستخدم');
        }
    }

    // ---------------------------------------------------------------------------------------------------- //
    // --------------------- Generate uniqe account id to be given to the company ------------------------- //
    // ---------------------------------------------------------------------------------------------------- //
    private function generateTruckingCompanyAccountId($company_id)
    {
        $sqlQuery = "SELECT DISTINCT (t.company_id + 502000) account_id
                     FROM trucking_company_view t
                     where t.company_id = ? ";

        $param = [$company_id];
        $result = DBConnection::runBindDatabaseQuery($sqlQuery, $param);

        if ($result[0]) {
            return $result[0]->account_id;
        } else {
            throw new Exception('لا يمكن انشاء حساب جديد للشركة ، حيث لم يتم تعريفها كشركة نقل');
        }
    }

    // ---------------------------------------------------------------------------------------------------- //
    // --------------------- Generate uniqe account id to be given to the company ------------------------- //
    // ---------------------------------------------------------------------------------------------------- //
    private function generateCompanyAccountId()
    {
        $sqlQuery = "SELECT
                    max(id ) + 1 account_id
                    FROM
                    account
                    where id > 501000
                    and id < 529999";

        $param = [];
        $result = DBConnection::runBindDatabaseQuery($sqlQuery, $param);

        if ($result[0]) {
            return $result[0]->account_id;
        } else {
            throw new Exception('لا يمكن انشاء حساب جديد ، لا يمكن انشاء رقم حساب');
        }
    }

    // ---------------------------------------------------------------------------------------- //
    // --------------------- validate account bean in order to create-------------------------- //
    // ---------------------------------------------------------------------------------------- //
    private function validateForCreate($accountBean, $update_by)
    {

        if ($accountBean->user_id) {
            $searchFilter = [['key' => "user_id", 'val' => $accountBean->user_id]];
        }
        if ($accountBean->company_id) {
            $searchFilter = [['key' => "company_id", 'val' => $accountBean->company_id]];
        }

        $account_qry = $this->searchAccount($searchFilter, 1, 0, $update_by);
        if ($account_qry->found_rows >= 1) {
            throw new Exception('الحساب موجود مسبقا', 555);
        }
    }


    // ------------------------------------------------------------------------- //
    // ------------------- change status of account ---------------------------- //
    // ------------------------------------------------------------------------- //
    public function changeStatus($account_id, $new_status, $user_id)
    {

        // if the arg is account_id or account bean
        if (gettype($account_id) == "object") {
            $accountBean = $account_id;
        } else {
            $accountBean = $this->getAccountBasic($account_id, $user_id);
        }

        // make sure the account balance is 0
        if ($accountBean->balance != 0) {
            throw new Exception('لا تستطيع المتابعة ، رصيد الحساب لا يساوي 0');
        }

        // change status
        $updateStruct = new stdClass();
        $updateStruct->id =  $accountBean->id . "," . $accountBean->sub_id;
        $updateStruct->status = $new_status;
        DBConnection::updateDB("account", $updateStruct, $user_id);
    }


    public function getTrialBalance($account_id, $create_date)
    {

        $query = "select  ifnull(jv_trx.current_balance,0) balance,
                            a.id , a.sub_id , a.type , a.tree_type , a.name ,  a.currency , a.cat
                    from 	account_view a left join jv_trx
                        on a.id=jv_trx.account_id
                        and a.sub_id=jv_trx.sub_id
                        and jv_trx.id in(
                            select max(id)
                            from jv_trx
                            where account_id=?
                                and create_date<?
                            group by sub_id)
                    where 	a.id=?
                    and a.sub_id>0
                    and a.create_date<?
                    order by cat desc,tree_type";
        $param = [$account_id, $create_date, $account_id, $create_date];
        $accountResult = DBConnection::runBindDatabaseQuery($query, $param);

        return $accountResult;
    }

    // -------------------------------------------------------------------------- //
    // -------------------search for searchCashBoxLog using any search filter ----------- //
    // -------------------------------------------------------------------------- //
    public function searchCashBoxLog($searchFilter, $limit, $offset, $user_id, $orderBy = null)
    {
        $searchAccountResult = DBConnection::searchDB("cash_box_log", $searchFilter, $limit, $offset, $user_id, $orderBy);
        return $searchAccountResult;
    }

}
