<?php

// import objects
require_once(dirname(__FILE__) . "/../../includes/DBConnection.php");
require_once(dirname(__FILE__) . "/../../includes/util.php");
require_once(dirname(__FILE__) . "/../waybill/waybill_core.php");
require_once(dirname(__FILE__) . "/../tender/tender_core.php");
require_once(dirname(__FILE__) . "/../payment/payment_core.php");
require_once(dirname(__FILE__) . "/../route_wage/route_wage_core.php");
require_once(dirname(__FILE__) . "/../route_wage/route_wage_core.php");
require_once(dirname(__FILE__) . "/../outgoing_integration/Jo_Petrol.php");

class DBR_Core
{

    public function __construct()
    {
        DBConnection::getInstance();
    }

    // --------------------------------------------------------------------------------------------------------------- //
    // ----------- Calculate DBR for certain truck using its waybill ------------------------------------------------- //
    // --------------------------------------------------------------------------------------------------------------- //
    public function calculateWaybillDBR($tn, $tender_id)
    {

        // init needed objects
        $waybillCore = new WaybillCore();
        $paymentCore = new PaymentCore();
        $routeWageCore = new RouteWageCore();

        // percentage of full dbr (0.9 = 90% of the full dbr will be returned)
        $dbr_percentage = 0.9;
        // The number of months from which the system calculates trips
        $start_period_months = 6;

        // get the  completed waybills for the truck on the selected tender in the past x months
        $before_six_months_date = date('Y-m-d', strtotime("-$start_period_months month"));

        $waybillFilter = [
            ['key' => 'tn', 'val' => $tn],
            ['key' => 'tender_id', 'val' => $tender_id],
            ['key' => 'status', 'val' => ['COMPLETE'], 'op' => "in"],
            ['key' => 'create_date', 'val' => $before_six_months_date, 'op' => 'date greater than'],
        ];
        $waybills = $waybillCore->searchWaybills($waybillFilter, 1000, 0, 0);

        if ($waybills->found_rows == 0) {
            $result = [];
            $result['number_of_wayblls'] = $waybills->found_rows;
            $result['dbr_value'] = 0;
            return $result;
        }

        // collect discharge weights and loss_in_kg;
        $discharge_weights = [];
        $loss_in_kgs = [];

        foreach ($waybills->data as $wbl) {
            $doc = json_decode($wbl->document);
            $discharge_weights[$wbl->wn] = $doc->cargo[0]->weights->discharge->net_weight;
            $loss_in_kgs[$wbl->wn] = $doc->freight->amount->loss_in_kg;
        }


        // sort discharge weights and loss_in_kg;
        asort($discharge_weights); // sort asc
        arsort($loss_in_kgs);      // sort desc


        // get the second maximum fine and the second lowest net_amount
        if (sizeof($discharge_weights) > 1) {
            $second_min_discgarge_weight = array_values($discharge_weights)[1];
        } else {
            $second_min_discgarge_weight = array_values($discharge_weights)[0];
        }

        $max_loss_in_kg = array_values($loss_in_kgs)[0];
        if ($max_loss_in_kg < 0) $max_loss_in_kg = 0;
        $discharge_weight_wn = array_keys($discharge_weights)[0];
        $loss_in_kg_wn = array_keys($loss_in_kgs)[0];

        // get the route wage object
        switch ($tender_id) {
            case '11':
                $origin_id = "91090021"; //مستودعات المصفاة في العقبة
                $destination_id = "91030015"; // مصفاة البترول في الزرقاء
                $cargo_id = "2010000003"; //النفط الخام
                $loading_weight = $second_min_discgarge_weight;
                $discharge_weight = $loading_weight - $max_loss_in_kg;
                $loading_date = date('Y-m-d', strtotime(DBConnection::getSystemDate()));
                $routeWage =  $routeWageCore->searchWageForFreight($tender_id, $origin_id, $destination_id, $cargo_id, $loading_date, 'payable', true, null);
                break;

            default:
                throw new Exception("calculate DBR for tender $tender_id is not supported");
                break;
        }

        // make virtual tagreesh for the folowing inputs
        $waybillBean = new stdClass();
        $waybillBean->document = new stdClass();
        $waybillBean->document->freight = null;
        $waybillBean->document->tender = new stdClass();
        $waybillBean->document->tender->id = $tender_id;
        $waybillBean->tender_id = $tender_id;
        $waybillBean->destination_id = $destination_id;
        $waybillBean->document->negotiable_instructios = new stdClass();
        $waybillBean->document->negotiable_instructios->route = new stdClass();
        $waybillBean->document->negotiable_instructios->route->destination = new stdClass();
        $waybillBean->document->negotiable_instructios->route->destination->id = $destination_id;


        try {
            $frieght_node = $paymentCore->calculateFreight($waybillBean, $discharge_weight, $loading_weight, $routeWage, null);
        } catch (Exception $ex) {
            $result = [];
            $result['number_of_wayblls'] = $waybills->found_rows;
            $result['result'] = "قيمة النقص المغرم كبيرة جدا";
            $result['error'] = $ex;
            $result['dbr_value'] = 0;
            return $result;
        }

        $loss_fine = $frieght_node->amount->loss_fine;

        // calculate trip per month factor (TPM)
        // get the number of trips in the past x months, if more than (number of months) then the factor = 1       
        $TPM = $waybills->found_rows >= $start_period_months ? 1 : ($waybills->found_rows / $start_period_months);

        // calculate late factor in the latest 3 trips
        $latestWaybillsFilter = [
            ['key' => 'tn', 'val' => $tn],
            ['key' => 'tender_id', 'val' => $tender_id],
            ['key' => 'status', 'val' => ['COMPLETE'], 'op' => "in"]
        ];
        $latestWaybills = $waybillCore->searchWaybills($latestWaybillsFilter, 3, 0, 0, ' order by id desc');

        $complete_durations = [];
        foreach ($latestWaybills->data as $latestWaybill) {
            $waybillActivity = $waybillCore->getWaybill($latestWaybill->id, 0);
            $waybillActivity = json_decode($waybillActivity)->activity;
            $complete_date = null;

            foreach ($waybillActivity as $act) {
                if ($act->object_new_status_code == "COMPLETE" && $act->action_code == "CHANGE_STATUS") {
                    $complete_date = $act->activity_date;
                }
            }

            $timeFirst  = strtotime($latestWaybill->discharge_date);
            $timeSecond = strtotime($complete_date);
            $differenceInSeconds = $timeSecond - $timeFirst;
            $hours = number_format($differenceInSeconds / 3600, 3);
            if ($hours < 0) $hours = 0;
            $complete_durations[] = $hours;
        }

        $avg_complete_waybill_in_hours = array_sum($complete_durations) / count($complete_durations);


        $late_factor = 48 / (47 + $avg_complete_waybill_in_hours);
        if ($late_factor > 1) $late_factor = 1;

        // apply all inputs in the formula
        $dbr_value = $frieght_node->amount->net_amount * $dbr_percentage * $TPM * $late_factor;

        // round the value to 10
        $dbr_value = round($dbr_value, -1);

        // format result
        $result = [];
        $result['number_of_wayblls'] = $waybills->found_rows;
        $result['waybill_date_start_period'] = $before_six_months_date;
        $result['discharge_weight'] = $discharge_weight;
        $result['discharge_weight_wn'] = $discharge_weight_wn;

        $result['loss_in_kg'] = $max_loss_in_kg;
        $result['loss_fine'] = $loss_fine;
        $result['loss_in_kg_wn'] = $loss_in_kg_wn;

        $result['tagreesh_net_amount'] = $frieght_node->amount->net_amount;
        $result['TPM'] = $TPM;

        $result['avg_complete_waybill_in_hours'] = number_format($avg_complete_waybill_in_hours, 3);
        $result['complete_durations'] = $complete_durations;

        $result['late_factor'] = number_format($late_factor, 3);
        $result['wage_per_ton'] = $routeWage->wage_per_ton;
        $result['wage_per_ton'] = $routeWage->wage_per_ton;
        $result['tagreesh_total_deductions'] = number_format($frieght_node->amount->total_deductions, 3);
        $result['dbr_value'] = $dbr_value;

        return $result;
    }


    // ------------------------------------------------------------------------------- //
    // ----------- Calculate the amount to be paid for a certain truck --------------- //
    // ------------------------------------------------------------------------------- //
    public function calculateTruckDownPayment($tn, $tender_id)
    {

        $tenderCore = new TenderCore();

        // get tender bean to get the maximum allowed amount for any truck
        $tenderBean = $tenderCore->getTenderBasic($tender_id, 0);
        if ($tenderBean->id == 11) {
            $max_limit = 400;
        } else {
            throw new Exception("calculate truck downpayment is not supported for this tender $tender_id");
        }

        // calculate the DBR for the truck
        $dbr = $this->calculateWaybillDBR($tn, $tender_id);

        // call FPS system to caculate how much the system can send him
        $fps = new FPS();
        $filterObj = new stdClass();
        $filterObj->filter = new stdClass();
        $filterObj->filter->tn = $tn;
        $filterObj->filter->status = "ACTIVE";
        $claimsReuslt =  $fps->searchClaims($filterObj, 0);

        $totalClaimRemaingAmount = 0;
        foreach ($claimsReuslt['data'] as $claim) {
            $remainingAmount =  $fps->getClaimRemainingAmount($claim['id'], 0);
            $totalClaimRemaingAmount += $remainingAmount;
        }
        $open_claims = ["value" => $totalClaimRemaingAmount];

        // find the final value
        if ($dbr['dbr_value'] + $open_claims['value'] > $max_limit) {
            $diff = ($dbr['dbr_value'] + $open_claims['value']) - $max_limit;
            $down_payment_value = $dbr['dbr_value'] - $diff;
        } else {
            $down_payment_value = $dbr['dbr_value'];
        }
        if ($down_payment_value < 0) $down_payment_value = 0;
        $down_payment = ["value" => $down_payment_value];

        // return result
        $result = [];
        $result['DBR'] = $dbr;
        $result['maximum_limit'] = $max_limit;
        $result['open_claims'] = $open_claims;
        $result['down_payment'] = $down_payment;
        return $result;
    }
}
